import {
    FastifyInstance,
    FastifyPluginOptions,
    FastifyPluginAsync
} from 'fastify';
import IPrisma from '@prisma/client'
import { defaultResponseApi, genErrorResponse, genInternalErrorResponse, prisma } from '../utils/utils';

interface GetWaypoint {
    ownerId: string
}

interface CreateWaypoint {
    ownerId: string
    name: string
    x: number
    y: number
    z: number
    dimension: string
    isPublic: boolean
}

interface UpdateWaypoint {
    id: number
    name?: string
    x?: number
    y?: number
    z?: number
    dimension?: string
    isPublic?: boolean
}

interface DeleteWaypoint {
    id: number
}

const WaypointRoute: FastifyPluginAsync = async (server: FastifyInstance, options: FastifyPluginOptions) => {
    server.get('/waypoints', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const waypoint = await prisma.wayPoint.findMany({
                include: {
                    User: true,
                },
                where: {
                    isPublic: true
                }
            })
            if (!waypoint) {
                response = genErrorResponse("Waypoint Empty")
            }
            response.result = waypoint
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.get<{ Params: GetWaypoint }>('/waypoint/:ownerId', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const waypoint = await prisma.wayPoint.findMany({
                include: {
                    User: true,
                },
                where: { ownerId: request.params.ownerId }
            })
            if (!waypoint) {
                response = genErrorResponse("Waypoint not found", 1001)
            }
            response.result = waypoint
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.post<{ Body: CreateWaypoint }>('/waypoint', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const existwaypoint = await prisma.wayPoint.findMany({
                where: {
                    name: request.body.name
                }
            })
            if (existwaypoint.length == 0) {
                const waypointCreate: CreateWaypoint = request.body
                const waypoint = await prisma.wayPoint.create({
                    data: waypointCreate
                })
                response.result = waypoint
            } else {
                response = genErrorResponse("Waypoint Name: " + request.body.name + " Already Exist", 1001)
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.put<{ Body: UpdateWaypoint }>('/waypoint', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const waypoint = await prisma.wayPoint.findFirst({
                where: {
                    id: request.body.id
                }
            })
            if (!waypoint) {
                return genErrorResponse("Waypoint ID: " + request.body.id + " Not Found", 1009)
            }

            const existName = await prisma.wayPoint.findFirst({
                where: {
                    name: request.body.name
                }
            })
            const waypointName = request.body.name
            if (existName && waypointName && (waypointName != waypoint.name)) {
                return genErrorResponse("Waypoint name: " + existName.name + " is exist")
            }

            const waypointUpdate = {
                name: request.body.name ?? waypoint.name,
                x: request.body.x ?? waypoint.x,
                y: request.body.y ?? waypoint.y,
                z: request.body.z ?? waypoint.z,
                dimension: request.body.dimension ?? waypoint.dimension,
                isPublic: request.body.isPublic ?? waypoint.isPublic,
            }
            const update = await prisma.wayPoint.update({
                data: waypointUpdate,
                where: {
                    id: request.body.id
                }
            })
            response.result = update

            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.delete<{ Params: DeleteWaypoint }>('/waypoint/:id', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const waypoint = await prisma.wayPoint.findFirst({
                where: {
                    id: Number(request.params.id)
                }
            })
            if (waypoint) {
                const deletewaypoint = await prisma.wayPoint.delete({
                    where: {
                        id: Number(request.params.id)
                    }
                })
                response.result = deletewaypoint
            } else {
                response = genErrorResponse("Waypoint ID: " + request.params.id + " Not Found")
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });
};

export default WaypointRoute;