import {
    FastifyInstance,
    FastifyPluginOptions,
    FastifyPluginAsync
} from 'fastify';
import IPrisma from '@prisma/client'
import { defaultResponseApi, genErrorResponse, genInternalErrorResponse, prisma } from '../utils/utils';

interface GetWarp {
    id: number
}

interface CreateWarp {
    name: string
    texture: string
    gamemode: string
    x: number
    y: number
    z: number
    active: boolean
}

interface UpdateWarp {
    id: number
    name?: string
    texture?: string
    x?: number
    y?: number
    z?: number
    active?: boolean
}

interface DeleteWarp {
    id: number
}

const WarpRoute: FastifyPluginAsync = async (server: FastifyInstance, options: FastifyPluginOptions) => {
    server.get('/warps', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const warp = await prisma.warp.findMany()
            if (!warp) {
                response = genErrorResponse("Warp Empty")
            }
            response.result = warp
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.get<{ Params: GetWarp }>('/warp/:id', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const warp = await prisma.warp.findFirst({
                where: { id: Number(request.params.id) }
            })
            if (!warp) {
                response = genErrorResponse("Warp not found")
            }
            response.result = warp
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.post<{ Body: CreateWarp }>('/warp', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const existwarp = await prisma.warp.findMany({
                where: {
                    name: request.body.name
                }
            })
            if (existwarp.length == 0) {
                const warpCreate: CreateWarp = request.body
                const warp = await prisma.warp.create({
                    data: warpCreate
                })
                response.result = warp
            } else {
                response = genErrorResponse("Warp Name: " + request.body.name + " Already Exist")
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.put<{ Body: UpdateWarp }>('/warp', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const existName = await prisma.warp.findFirst({
                where: {
                    name: request.body.name
                }
            })
            if (existName && (request.body.name != existName.name)) {
                return genErrorResponse("Warp name: " + existName.name + " is exist")
            }
            const warp = await prisma.warp.findFirst({
                where: {
                    id: request.body.id
                }
            })
            if (warp) {
                const warpUpdate = {
                    name: request.body.name ?? warp.name,
                    texture: request.body.texture ?? warp.texture,
                    x: request.body.x ?? warp.x,
                    y: request.body.y ?? warp.y,
                    z: request.body.z ?? warp.z,
                    active: request.body.active ?? warp.active,
                }
                const update = await prisma.warp.update({
                    data: warpUpdate,
                    where: {
                        id: request.body.id
                    }
                })
                response.result = update
            } else {
                response = genErrorResponse("Warp ID: " + request.body.id + " Not Found")
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.delete<{ Params: DeleteWarp }>('/warp/:id', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const warp = await prisma.warp.findFirst({
                where: {
                    id: Number(request.params.id)
                }
            })
            if (warp) {
                const deletewarp = await prisma.warp.delete({
                    where: {
                        id: Number(request.params.id)
                    }
                })
                response.result = deletewarp
            } else {
                response = genErrorResponse("Warp ID: " + request.params.id + " Not Found")
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });
};

export default WarpRoute;