import {
    FastifyInstance,
    FastifyPluginOptions,
    FastifyPluginAsync
} from 'fastify';
import IPrisma from '@prisma/client'
import { defaultResponseApi, genErrorResponse, genInternalErrorResponse, prisma } from '../utils/utils';

interface GetUser {
    id: string
}

interface CreateUser {
    xuid: string
    name: string
    role: IPrisma.UserRole
}

interface UpdateUser {
    xuid: string
    name?: string
    canTpa?: boolean
    role?: IPrisma.UserRole
}

interface DeleteUser {
    xuid: string
}

const UserRoute: FastifyPluginAsync = async (server: FastifyInstance, options: FastifyPluginOptions) => {
    server.get('/users', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const user = await prisma.user.findMany()
            if (!user) {
                response = genErrorResponse("User Empty")
            }
            response.result = user
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.get<{ Params: GetUser }>('/user/:id', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const user = await prisma.user.findFirst({
                include: {
                    guildMember: true,
                },
                where: { xuid: request.params.id }
            })
            if (!user) {
                response = genErrorResponse("User not found")
            }

            response.result = user

            if (user?.guildMember) {
                const guild = await prisma.guild.findFirst({ where: { guildId: user.guildId! } })
                response.result.guild = guild
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.post<{ Body: CreateUser }>('/user', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const existUser = await prisma.user.findMany({
                where: {
                    OR: [
                        { xuid: request.body.xuid },
                    ]
                }
            })
            if (existUser.length == 0) {
                const user = await prisma.user.create({
                    data: {
                        xuid: request.body.xuid,
                        name: request.body.name,
                        role: IPrisma.UserRole.USER
                    }
                })
                response.result = user
            } else {
                const user = await prisma.user.update({
                    data: {
                        name: request.body.name,
                    },
                    where: {
                        xuid: request.body.xuid,
                    }
                })
                response.result = user
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.put<{ Body: UpdateUser }>('/user', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const user = await prisma.user.findFirst({
                where: {
                    xuid: request.body.xuid
                }
            })
            if (user) {
                const update = await prisma.user.update({
                    data: {
                        name: request.body.name ?? user.name,
                        role: request.body.role ?? user.role,
                        canTpa: request.body.canTpa ?? user.canTpa
                    },
                    where: {
                        xuid: request.body.xuid
                    }
                })
                response.result = update
            } else {
                response = genErrorResponse("User XUID: " + request.body.xuid + " Not Found")
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.delete<{ Params: DeleteUser }>('/user/:xuid', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const user = await prisma.user.findFirst({
                where: {
                    xuid: request.params.xuid
                }
            })
            if (user) {
                const deleteUser = await prisma.user.delete({
                    where: {
                        xuid: request.params.xuid
                    }
                })
                response.result = deleteUser
            } else {
                response = genErrorResponse("User XUID: " + request.params.xuid + " Not Found")
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });
};

export default UserRoute;