import {
    FastifyInstance,
    FastifyPluginOptions,
    FastifyPluginAsync
} from 'fastify';
import IPrisma from '@prisma/client'
import { defaultResponseApi, genErrorResponse, genInternalErrorResponse, prisma } from '../utils/utils';

interface GetTransaction {
    xuid: string
}

interface TransferMoney {
    xuid: string
    targetName: string
    amount: number
}

interface GiftMoney {
    xuid: string
    amount: number
}

interface DeleteHistory {
    xuid: string
}

const TransactionRoute: FastifyPluginAsync = async (server: FastifyInstance, options: FastifyPluginOptions) => {
    server.get<{ Params: GetTransaction }>('/transaction/:xuid', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const xuid = request.params.xuid

            const user = await prisma.user.findFirst({ where: { xuid } })
            if (!user) {
                return genErrorResponse("User XUID: " + xuid + " not found", 1001)
            }

            const transaction = await prisma.transaction.findMany({
                include: {
                    TransactionDetail: {
                        include: {
                            shopItem: true
                        }
                    },
                    targetUser: true
                },
                where: {
                    userId: xuid
                }
            })

            response.result = transaction
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.post<{ Body: TransferMoney }>('/transaction/transfer', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const xuid = request.body.xuid
            const targetName = request.body.targetName
            const amount = request.body.amount

            const transaction = await prisma.$transaction(async (tx) => {
                const userFrom = await tx.user.findFirst({ where: { xuid } })
                if (!userFrom) {
                    return genErrorResponse("User Source ID: " + xuid + " not found", 1001)
                }

                const userTarget = await tx.user.findFirst({ where: { name: targetName } })
                if (!userTarget) {
                    return genErrorResponse("User Target Name: " + targetName + " not found", 1002)
                }

                if (userFrom.money < amount) {
                    return genErrorResponse("Money user: " + userFrom.money + " not enough, need " + amount, 1003)
                }

                const transferItem = await tx.shopItem.findFirst({ where: { name: 'TRANSFER' } })
                if (!transferItem) {
                    return genErrorResponse("Transfer item not available", 1004)
                }

                const transaction1 = await tx.transaction.create({
                    data: {
                        userId: xuid,
                        targetUserId: userTarget.xuid,
                        trxType: IPrisma.TrxType.DEBIT,
                    }
                })

                const transaction2 = await tx.transaction.create({
                    data: {
                        userId: userTarget.xuid,
                        targetUserId: xuid,
                        trxType: IPrisma.TrxType.CREDIT,
                    }
                })

                await tx.transactionDetail.create({
                    data: {
                        transactionId: transaction1.transactionId,
                        itemId: transferItem.itemId,
                        qty: amount,
                        totalPrice: amount
                    }
                })

                await tx.transactionDetail.create({
                    data: {
                        transactionId: transaction2.transactionId,
                        itemId: transferItem.itemId,
                        qty: amount,
                        totalPrice: amount
                    }
                })

                await tx.user.update({
                    data: {
                        money: userFrom.money - amount
                    },
                    where: {
                        xuid
                    }
                })

                await tx.user.update({
                    data: {
                        money: userTarget.money + amount
                    },
                    where: {
                        xuid: userTarget.xuid
                    }
                })

                return transaction1
            })

            response.result = transaction
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.post<{ Body: GiftMoney }>('/transaction/gift', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const xuid = request.body.xuid
            const amount = request.body.amount

            const transaction = await prisma.$transaction(async (tx) => {
                const userFrom = await tx.user.findFirst({ where: { xuid } })
                if (!userFrom) {
                    return genErrorResponse("User Source ID: " + xuid + " not found", 1001)
                }

                const giftItem = await tx.shopItem.findFirst({ where: { name: 'GIFT' } })
                if (!giftItem) {
                    return genErrorResponse("Gift item not available", 1001)
                }

                const transaction_user = await tx.transaction.create({
                    data: {
                        userId: xuid,
                        trxType: IPrisma.TrxType.CREDIT,
                    }
                })

                await tx.transactionDetail.create({
                    data: {
                        transactionId: transaction_user.transactionId,
                        itemId: giftItem.itemId,
                        qty: amount,
                        totalPrice: amount
                    }
                })

                await tx.user.update({
                    data: {
                        money: userFrom.money + amount
                    },
                    where: {
                        xuid
                    }
                })


                return transaction_user
            })

            response.result = transaction
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.delete<{ Body: DeleteHistory }>('/transaction', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const xuid = request.body.xuid

            const transaction = await prisma.$transaction(async (tx) => {
                const userFrom = await tx.user.findFirst({ where: { xuid } })
                if (!userFrom) {
                    return genErrorResponse("User Source ID: " + xuid + " not found", 1001)
                }

                const deleteAll = await tx.transaction.deleteMany({
                    where: {
                        userId: xuid
                    }
                })   

                return deleteAll
            })

            response.result = transaction
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });
}

export default TransactionRoute;