import {
    FastifyInstance,
    FastifyPluginOptions,
    FastifyPluginAsync
} from 'fastify';
import IPrisma from '@prisma/client'
import { defaultResponseApi, genErrorResponse, genInternalErrorResponse, prisma } from '../utils/utils';

interface GetListTpa {
    name: string
}

interface GetSingleTpa {
    fromName: string
    targetName: string
}

interface CreateTpa {
    fromName: string
    targetName: string
}

interface UpdateTpa {
    id: number
    fromName?: string
    targetName?: string
}

interface DeleteTpa {
    fromName: string
    targetName: string
}

const TpaRoute: FastifyPluginAsync = async (server: FastifyInstance, options: FastifyPluginOptions) => {
    server.get('/tpas', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const tpa = await prisma.tpaRequest.findMany()
            if (!tpa) {
                response = genErrorResponse("Tpa Empty")
            }
            response.result = tpa
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.get<{ Querystring: GetSingleTpa }>('/tpa/request', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const tpa = await prisma.tpaRequest.findFirst({
                where: {
                    AND: [
                        { fromName: request.query.fromName },
                        { targetName: request.query.targetName },
                    ]
                }
            })
            if (!tpa) {
                response = genErrorResponse("Tpa not found", 1001)
            }
            response.result = tpa
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.get<{ Querystring: GetListTpa }>('/tpa/list', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const tpa = await prisma.tpaRequest.findMany({
                where: {
                    OR: [
                        { fromName: request.query.name },
                        { targetName: request.query.name },
                    ]
                }
            })
            if (!tpa) {
                response = genErrorResponse("Tpa not found")
            }
            response.result = tpa
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.post<{ Body: CreateTpa }>('/tpa', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const existtpa = await prisma.tpaRequest.findMany({
                where: {
                    AND: [
                        {
                            fromName: request.body.fromName
                        },
                        {
                            targetName: request.body.targetName
                        }
                    ]
                }
            })
            if (existtpa.length == 0) {
                const tpaCreate: CreateTpa = request.body
                const tpa = await prisma.tpaRequest.create({
                    data: tpaCreate
                })
                response.result = tpa
            } else {
                response = genErrorResponse("Tpa Request from " + request.body.fromName + " to " + request.body.targetName + " already exist", 401)
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.put<{ Body: UpdateTpa }>('/tpa', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const existName = await prisma.tpaRequest.findFirst({
                where: {
                    AND: [
                        {
                            fromName: request.body.fromName
                        },
                        {
                            targetName: request.body.targetName
                        }
                    ]
                }
            })
            if (existName && (request.body.fromName != existName.fromName) && (request.body.targetName != existName.targetName)) {
                return genErrorResponse("Tpa Request from " + request.body.fromName + " to " + request.body.targetName + " already exist", 401)
            }
            const tpa = await prisma.tpaRequest.findFirst({
                where: {
                    id: request.body.id
                }
            })
            if (tpa) {
                const tpaUpdate = {
                    fromName: request.body.fromName ?? tpa.fromName,
                    targetName: request.body.targetName ?? tpa.targetName,
                }
                const update = await prisma.tpaRequest.update({
                    data: tpaUpdate,
                    where: {
                        id: request.body.id
                    }
                })
                response.result = update
            } else {
                response = genErrorResponse("Tpa ID: " + request.body.id + " Not Found")
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.delete<{ Body: DeleteTpa }>('/tpa', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const tpa = await prisma.tpaRequest.findFirst({
                where: {
                    AND: [
                        { fromName: request.body.fromName },
                        { targetName: request.body.targetName },
                    ]
                }
            })
            if (tpa) {
                const deletetpa = await prisma.tpaRequest.deleteMany({
                    where: {
                        AND: [
                            { fromName: request.body.fromName },
                            { targetName: request.body.targetName },
                        ]
                    }
                })
                response.result = deletetpa
            } else {
                response = genErrorResponse("Tpa Request Not Found", 1001)
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });
};

export default TpaRoute;