import {
    FastifyInstance,
    FastifyPluginOptions,
    FastifyPluginAsync
} from 'fastify';
import IPrisma from '@prisma/client'
import { defaultResponseApi, genErrorResponse, genInternalErrorResponse, prisma } from '../utils/utils';

interface CreateCategory {
    no: number
    name: string
    texture: string
    type?: IPrisma.ItemType
    active?: boolean
}

interface UpdateCategory {
    id: number
    no: number
    name?: string
    texture?: string
    type?: IPrisma.ItemType
    active?: boolean
}

interface DeleteCategory {
    id: number
}

interface GetItem {
    id: number
}

interface ParamsGetItem {
    page?: number
    limit?: number
    category?: number
}

interface CreateItem {
    categoryId: number
    texture: string
    name: string
    cost: number
    sell: number
    data: number
    item: string
    maxLevel?: number
    enchant?: string
    enchantType?: IPrisma.EnchantType
}

interface UpdateItem {
    id: number
    categoryId?: number
    texture?: string
    name?: string
    cost?: number
    sell?: number
    data?: number
    item?: string
    maxLevel?: number
    enchant?: string
    enchantType?: IPrisma.EnchantType
}

interface DeleteItem {
    id: number
}

interface BuyItem {
    xuid: string
    itemId: number
    qty: number
}

interface SellItem {
    xuid: string
    itemId: number
    qty: number
}

const ShopRoute: FastifyPluginAsync = async (server: FastifyInstance, options: FastifyPluginOptions) => {

    // [ CATEGORY ]

    server.get('/shop/category', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const category = await prisma.shopCategory.findMany({
                select: {
                    categoryID: true, no: true, name: true, texture: true, type: true
                },
                orderBy: {
                    no: 'asc'
                },
                where: {
                    active: true
                }
            })
            if (!category) {
                response = genErrorResponse("Category not found")
            }
            response.result = category
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.post<{ Body: CreateCategory }>('/shop/category', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const category = await prisma.shopCategory.create({
                data: request.body
            })
            response.result = category
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.put<{ Body: UpdateCategory }>('/shop/category', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const category = await prisma.shopCategory.findFirst({
                where: {
                    categoryID: request.body.id
                }
            })
            if (category) {
                const categoryUpdate = {
                    no: request.body.no ?? category.no,
                    name: request.body.name ?? category.name,
                    texture: request.body.texture ?? category.texture,
                    type: request.body.type ?? category.type
                }
                const update = await prisma.shopCategory.update({
                    data: categoryUpdate,
                    where: {
                        categoryID: request.body.id
                    }
                })
                response.result = update
            } else {
                response = genErrorResponse("Category Not Found")
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.delete<{ Params: DeleteCategory }>('/shop/category/:id', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const category = await prisma.shopCategory.findFirst({
                where: {
                    categoryID: Number(request.params.id)
                }
            })
            if (category) {
                const deleteCategory = await prisma.shopCategory.delete({
                    where: {
                        categoryID: Number(request.params.id)
                    }
                })
                response.result = deleteCategory
            } else {
                response = genErrorResponse("Category Not Found")
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    // [ ITEMS ]

    server.get<{ Querystring: ParamsGetItem }>('/shop/items', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            var item = {}

            if (request.query.page) {
                const limit = request.query.limit ?? 15
                const page = request.query.page

                item = await prisma.shopItem.findMany({
                    skip: limit * (page - 1),
                    take: limit
                })
            } else if (request.query.category) {
                item = await prisma.shopItem.findMany({
                    where: { categoryId: Number(request.query.category) }
                })
            } else {
                item = await prisma.shopItem.findMany()
            }

            if (!item) {
                response = genErrorResponse("Item not found")
            }
            response.result = item
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.get<{ Params: GetItem }>('/shop/item/:id', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            var item = await prisma.shopItem.findFirst({ where: { itemId: Number(request.params.id) } })
            if (!item) {
                response = genErrorResponse("Item not found", 1001)
            }
            response.result = item
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.get('/shop/item/exp', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            var item = await prisma.shopItem.findFirst({ where: { name: 'EXP' } })
            if (!item) {
                response = genErrorResponse("Item not found", 1001)
            }
            response.result = item
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.post<{ Body: CreateItem }>('/shop/item', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const item = await prisma.shopItem.create({
                data: request.body
            })
            response.result = item
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.put<{ Body: UpdateItem }>('/shop/item', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const item = await prisma.shopItem.findFirst({
                where: {
                    itemId: request.body.id
                }
            })
            if (item) {
                if (request.body.categoryId) {
                    const itemCategory = await prisma.shopCategory.findFirst({
                        where: {
                            categoryID: request.body.categoryId
                        }
                    })
                    if (!itemCategory) {
                        return genErrorResponse("Category with ID: " + request.body.categoryId + " not found")
                    }
                }
                const categoryUpdate = {
                    categoryId: request.body.categoryId ?? item.categoryId,
                    texture: request.body.texture ?? item.texture,
                    name: request.body.name ?? item.name,
                    cost: request.body.cost ?? item.cost,
                    sell: request.body.sell ?? item.sell,
                    data: request.body.data ?? item.data,
                    item: request.body.item ?? item.item,
                    maxLevel: request.body.maxLevel ?? item.maxLevel,
                    enchant: request.body.enchant ?? item.enchant,
                    enchantType: request.body.enchantType ?? item.enchantType,
                }
                const update = await prisma.shopItem.update({
                    data: categoryUpdate,
                    where: {
                        itemId: request.body.id
                    }
                })
                response.result = update
            } else {
                response = genErrorResponse("Item Not Found")
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.delete<{ Params: DeleteItem }>('/shop/item/:id', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const item = await prisma.shopItem.findFirst({
                where: {
                    itemId: Number(request.params.id)
                }
            })
            if (item) {
                const deleteItem = await prisma.shopItem.delete({
                    where: {
                        itemId: Number(request.params.id)
                    }
                })
                response.result = deleteItem
            } else {
                response = genErrorResponse("Item Not Found")
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    // [ TRX FLOW ]
    server.post<{ Body: BuyItem }>('/shop/buy', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const xuid = request.body.xuid
            const itemId = request.body.itemId
            const qty = request.body.qty

            const result = await prisma.$transaction(async (tx) => {
                const user = await tx.user.findFirst({ where: { xuid } })
                if (!user) {
                    return genErrorResponse("User ID: " + xuid + " not found", 1001)
                }

                const shopItem = await tx.shopItem.findFirst({ where: { itemId } })
                if (!shopItem) {
                    return genErrorResponse("Item ID: " + itemId + " not found", 1002)
                }

                const totalPrice = shopItem.cost * qty
                if (user.money < totalPrice) {
                    return genErrorResponse("Money user: " + user.money + " not enough, need " + totalPrice, 1003)
                }

                const transaction = await tx.transaction.create({
                    data: {
                        userId: xuid,
                        trxType: IPrisma.TrxType.DEBIT,
                    }
                })

                await tx.transactionDetail.create({
                    data: {
                        transactionId: transaction.transactionId,
                        itemId: shopItem.itemId,
                        qty,
                        totalPrice
                    }
                })

                await tx.user.update({
                    data: {
                        money: user.money - totalPrice
                    },
                    where: {
                        xuid
                    }
                })

                var resp = defaultResponseApi()
                resp.result = transaction
                return resp
            })

            response = result
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.post<{ Body: SellItem }>('/shop/sell', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const xuid = request.body.xuid
            const itemId = request.body.itemId
            const qty = request.body.qty

            const result = await prisma.$transaction(async (tx) => {
                const user = await tx.user.findFirst({ where: { xuid } })
                if (!user) {
                    return genErrorResponse("User ID: " + xuid + " not found", 1001)
                }

                const shopItem = await tx.shopItem.findFirst({ where: { itemId } })
                if (!shopItem) {
                    return genErrorResponse("Item ID: " + itemId + " not found", 1002)
                }

                if (!shopItem.sell) {
                    return genErrorResponse("Item ID: " + itemId + " is not sell", 1003)
                }

                const totalPrice = shopItem.sell * qty
                const transaction = await tx.transaction.create({
                    data: {
                        userId: xuid,
                        trxType: IPrisma.TrxType.CREDIT,
                    }
                })

                await tx.transactionDetail.create({
                    data: {
                        transactionId: transaction.transactionId,
                        itemId: shopItem.itemId,
                        qty,
                        totalPrice
                    }
                })

                await tx.user.update({
                    data: {
                        money: user.money + totalPrice
                    },
                    where: {
                        xuid
                    }
                })

                var resp = defaultResponseApi()
                resp.result = transaction
                return resp
            })

            response = result
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });
};

export default ShopRoute;