import {
    FastifyInstance,
    FastifyPluginOptions,
    FastifyPluginAsync
} from 'fastify';
import IPrisma from '@prisma/client'
import { defaultResponseApi, genErrorResponse, genInternalErrorResponse, prisma } from '../utils/utils';

interface GetSetting {
    id: number
}

interface CreateSetting {
    name: string
    value: string
    type: IPrisma.DataType
}

interface UpdateSetting {
    id: number
    name?: string
    value?: string
    type?: IPrisma.DataType
}

interface DeleteSetting {
    id: number
}

const SettingRoute: FastifyPluginAsync = async (server: FastifyInstance, options: FastifyPluginOptions) => {
    server.get('/settings', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const setting = await prisma.settings.findMany()
            if (!setting) {
                response = genErrorResponse("Setting Empty")
            }
            response.result = setting
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.get<{ Params: GetSetting }>('/setting/:id', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const setting = await prisma.settings.findFirst({
                where: { id: Number(request.params.id) }
            })
            if (!setting) {
                response = genErrorResponse("Setting not found")
            }
            response.result = setting
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.post<{ Body: CreateSetting }>('/setting', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const existsetting = await prisma.settings.findMany({
                where: {
                    name: request.body.name
                }
            })
            if (existsetting.length == 0) {
                const settingCreate: CreateSetting = request.body
                const setting = await prisma.settings.create({
                    data: settingCreate
                })
                response.result = setting
            } else {
                response = genErrorResponse("Setting Name: " + request.body.name + " Already Exist")
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.put<{ Body: UpdateSetting }>('/setting', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const existID = await prisma.settings.findFirst({
                where: {
                    id: request.body.id
                }
            })
            const existName = await prisma.settings.findFirst({
                where: {
                    name: request.body.name
                }
            })
            if (existName && (existName.name != existID?.name)) {
                return genErrorResponse("Setting name: " + existName.name + " is exist")
            }
            const setting = await prisma.settings.findFirst({
                where: {
                    id: request.body.id
                }
            })
            if (setting) {
                const settingUpdate = {
                    name: request.body.name ?? setting.name,
                    value: request.body.value ?? setting.value,
                    type: request.body.type ?? setting.type,
                }
                const update = await prisma.settings.update({
                    data: settingUpdate,
                    where: {
                        id: request.body.id
                    }
                })
                response.result = update
            } else {
                response = genErrorResponse("Setting ID: " + request.body.id + " Not Found")
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.delete<{ Params: DeleteSetting }>('/setting/:id', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const setting = await prisma.settings.findFirst({
                where: {
                    id: Number(request.params.id)
                }
            })
            if (setting) {
                const deletesetting = await prisma.settings.delete({
                    where: {
                        id: Number(request.params.id)
                    }
                })
                response.result = deletesetting
            } else {
                response = genErrorResponse("Setting ID: " + request.params.id + " Not Found")
            }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });
};

export default SettingRoute;