import {
    FastifyInstance,
    FastifyPluginOptions,
    FastifyPluginAsync
} from 'fastify';
import IPrisma from '@prisma/client'
import { defaultResponseApi, genErrorResponse, genInternalErrorResponse, prisma } from '../utils/utils';
import { SETTINGS } from '../constants/settings';

interface GetGuild {
    id: number
}

interface CreateGuild {
    xuid: string
    name: string
    description?: string
    maxMember: number
    isPublic?: boolean
}

interface UpdateGuild {
    xuid: string
    name?: string
    description?: string
    maxMember?: number
    isPublic?: boolean
}

interface JoinGuild {
    xuid: string
    guildId: number
}

interface PromoteGuild {
    xuid: string
    guildId: number
    role: IPrisma.GuildRole
}

interface LeaveGuild {
    xuid: string
}

interface DeleteGuild {
    xuid: string
}

const GuildRoute: FastifyPluginAsync = async (server: FastifyInstance, options: FastifyPluginOptions) => {

    const cost_member = Number((await prisma.settings.findFirst({ where: { name: SETTINGS.GUILD_CREATE_PRICE } }))?.value ?? 20000)

    server.get('/guild/cost_member', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            response.result = { cost_member }
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.get('/guilds', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const guild = await prisma.guild.findMany({
                where: { isPublic: true, },
                orderBy: { updatedAt: 'desc' }

            })
            if (!guild) {
                return genErrorResponse("Guild Empty")
            }
            response.result = guild
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.get<{ Params: GetGuild }>('/guild/:id', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const guild = await prisma.guild.findFirst({
                where: { guildId: Number(request.params.id) }
            })
            if (!guild) {
                return genErrorResponse("Guild ID " + request.params.id + " not found")
            }

            const guildMember = await prisma.guildMember.findMany({
                include: { member: true },
                where: { guildId: Number(request.params.id) }
            })
            if (!guildMember) {
                return genErrorResponse("Guild Member ID " + request.params.id + " not found")
            }
            response.result = guild
            response.result.members = guildMember
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.post<{ Body: CreateGuild }>('/guild/create', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const xuid = request.body.xuid
            const name = request.body.name
            const description = request.body.description
            const max_member = request.body.maxMember
            const is_public = request.body.isPublic

            const guild_create = await prisma.$transaction(async (tx) => {

                const owner = await tx.user.findFirst({ where: { xuid } })

                const guild_cost = max_member * cost_member

                if (owner?.money! < guild_cost) {
                    return genErrorResponse("Owner money is not enough, need " + guild_cost, 1001)
                }

                const hasGuild = await tx.guild.findFirst({
                    where: { ownerId: xuid }
                })
                if (hasGuild) {
                    return genErrorResponse("Owner still have guild " + hasGuild.name + "", 1002)
                }

                const guildExist = await tx.guild.findFirst({
                    where: { name }
                })
                if (guildExist) {
                    return genErrorResponse("Guild with name " + guildExist.name + " already exists", 1003)
                }

                if (name.length > 50) {
                    return genErrorResponse("Max length name is 50", 1004)
                }

                const guild = await tx.guild.create({
                    data: {
                        ownerId: xuid,
                        name,
                        description,
                        maxMembers: max_member,
                        isPublic: is_public
                    }
                })

                if (!guild) {
                    return genErrorResponse("Failed create guild", 1005)
                }

                await tx.user.update({
                    data: {
                        money: owner?.money! - guild_cost,
                        guildId: guild.guildId
                    },
                    where: {
                        xuid: owner?.xuid
                    }
                })

                await tx.guildMember.create({
                    data: {
                        guildId: guild.guildId,
                        memberId: xuid,
                        memberRole: IPrisma.GuildRole.LEADER
                    }
                })

                const guildItem = await tx.shopItem.findFirst({ where: { name: 'GUILD CREATE' } })
                if (!guildItem) {
                    return genErrorResponse("Guild item not available", 1006)
                }

                const guild_trx = await tx.transaction.create({
                    data: {
                        userId: xuid,
                        trxType: IPrisma.TrxType.DEBIT,
                    }
                })

                await tx.transactionDetail.create({
                    data: {
                        transactionId: guild_trx.transactionId,
                        itemId: guildItem.itemId,
                        qty: guild_cost,
                        totalPrice: guild_cost
                    }
                })

                return guild
            })

            response.result = guild_create
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.put<{ Body: UpdateGuild }>('/guild', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const xuid = request.body.xuid
            const name = request.body.name
            const description = request.body.description
            const max_member = request.body.maxMember
            const is_public = request.body.isPublic

            const guild_update = await prisma.$transaction(async (tx) => {

                const userUpdater = await tx.user.findFirst({ where: { xuid } })

                if (!userUpdater) {
                    return genErrorResponse("User XUID: " + xuid + " Not Found", 1001)
                }

                if (!userUpdater?.guildId) {
                    return genErrorResponse("User XUID: " + xuid + " Don\'t have any guild", 1002)
                }

                const guild_id = userUpdater.guildId

                const guild = await tx.guild.findFirst({
                    include: {
                        GuildMember: true
                    },
                    where: {
                        guildId: guild_id
                    }
                })

                const guild_member = await tx.guildMember.findFirst({ where: { memberId: xuid } })

                if (!guild) {
                    return genErrorResponse("Guild ID: " + guild_id + " Not Found", 1009)
                }

                if (guild_member?.memberRole == IPrisma.GuildRole.MEMBER) {
                    return genErrorResponse("XUID: " + xuid + " don't have access to edit this guild", 1003)
                }

                var maxChange = null
                var diffMax = 0

                if (max_member != null && guild.maxMembers < max_member) {
                    maxChange = "+"
                    diffMax = max_member - guild.maxMembers
                    var guild_cost = diffMax * cost_member
                    if (userUpdater?.money! < guild_cost) {
                        return genErrorResponse("User money is not enough to add guild max member", 1004)
                    }

                    const guildItem = await tx.shopItem.findFirst({ where: { name: 'GUILD UPDATE MAX' } })
                    if (!guildItem) {
                        return genErrorResponse("Guild item not available", 1007)
                    }

                    const guild_trx = await tx.transaction.create({
                        data: {
                            userId: xuid,
                            trxType: IPrisma.TrxType.DEBIT,
                        }
                    })

                    await tx.transactionDetail.create({
                        data: {
                            transactionId: guild_trx.transactionId,
                            itemId: guildItem.itemId,
                            qty: guild_cost,
                            totalPrice: guild_cost
                        }
                    })
                } else if (max_member != null && guild.maxMembers > max_member) {
                    maxChange = "-"
                    diffMax = guild.maxMembers - max_member
                    const guildLength = guild.GuildMember.length
                    var guild_cost = diffMax * cost_member
                    const cashback = (guild_cost / 2)

                    if (guildLength > max_member) {
                        return genErrorResponse("Failed update max member. Guild length is " + guildLength + " but you requesting max member to " + max_member + " ", 1005)
                    }

                    const guildItem = await tx.shopItem.findFirst({ where: { name: 'GUILD UPDATE MAX' } })
                    if (!guildItem) {
                        return genErrorResponse("Guild item not available", 1007)
                    }

                    const guild_trx = await tx.transaction.create({
                        data: {
                            userId: xuid,
                            trxType: IPrisma.TrxType.CREDIT,
                        }
                    })

                    await tx.transactionDetail.create({
                        data: {
                            transactionId: guild_trx.transactionId,
                            itemId: guildItem.itemId,
                            qty: cashback,
                            totalPrice: cashback
                        }
                    })
                }

                if ((name ?? guild.name).length > 50) {
                    return genErrorResponse("Max length name is 50", 1006)
                }

                const guildUpdate = {
                    name: name ?? guild.name,
                    description: description ?? guild.description,
                    maxMembers: max_member ?? guild.maxMembers,
                    isPublic: is_public ?? guild.isPublic
                }

                const update = await tx.guild.update({
                    data: guildUpdate,
                    where: {
                        guildId: guild_id
                    }
                })

                const guildCost = diffMax * cost_member

                if (maxChange == "+") {
                    await tx.user.update({
                        data: {
                            money: userUpdater?.money! - guildCost
                        },
                        where: {
                            xuid: userUpdater?.xuid
                        }
                    })
                } else if (maxChange == "-") {
                    await tx.user.update({
                        data: {
                            money: userUpdater?.money! + (guildCost / 2)
                        },
                        where: {
                            xuid: userUpdater?.xuid
                        }
                    })
                }

                return update
            })

            response.result = guild_update

            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.post<{ Body: JoinGuild }>('/guild/join', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const xuid = request.body.xuid
            const guild_id = request.body.guildId

            const guild = await prisma.guild.findFirst({
                include: {
                    GuildMember: true
                },
                where: {
                    guildId: guild_id
                }
            })

            const member = await prisma.user.findFirst({ where: { xuid } })
            if (member?.guildId) {
                return genErrorResponse("Cannot join. User still have guild with ID: " + member?.guildId, 1001)
            }

            if (!guild) {
                return genErrorResponse("Guild ID: " + guild_id + " Not Found", 1009)
            }

            if ((guild?.GuildMember?.length ?? 0) == guild?.maxMembers) {
                return genErrorResponse("Guild " + guild.name + " has reached max member", 1002)
            }

            const guildMember = await prisma.guildMember.create({
                data: {
                    guildId: guild_id,
                    memberId: xuid,
                }
            })

            if (!guildMember) {
                return genErrorResponse("Failed to join guild " + guild.name, 1003)
            }

            await prisma.user.update({
                data: {
                    guildId: guild.guildId
                },
                where: {
                    xuid
                }
            })

            response.result = guildMember
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.post<{ Body: PromoteGuild }>('/guild/promote', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const xuid = request.body.xuid
            const guild_id = request.body.guildId
            const role = request.body.role

            const guild = await prisma.guild.findFirst({
                include: {
                    GuildMember: true
                },
                where: {
                    guildId: guild_id
                }
            })

            if (!guild) {
                return genErrorResponse("Guild ID: " + guild_id + " Not Found", 1001)
            }

            console.log({ xuid, guild_id, role })
            if (guild.ownerId == xuid) {
                return genErrorResponse("Owner cannot demote", 1002)
            }

            const guildMember = await prisma.guildMember.update({
                data: {
                    memberRole: role
                },
                where: {
                    guildId: guild_id,
                    memberId: xuid,
                }
            })

            if (!guildMember) {
                return genErrorResponse("Failed to promote ID: " + xuid, 1002)
            }

            response.result = guildMember
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.post<{ Body: LeaveGuild }>('/guild/leave', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const xuid = request.body.xuid

            const member = await prisma.user.findFirst({ where: { xuid } })
            if (!member) {
                return genErrorResponse("User ID: " + xuid + " Not Exist", 1001)
            }

            if (!member?.guildId) {
                return genErrorResponse("Cannot leave. User don\'t have any guild", 1002)
            }

            const guild_id = member?.guildId

            const guild = await prisma.guild.findFirst({
                where: {
                    guildId: guild_id
                }
            })

            if (!guild) {
                return genErrorResponse("Guild ID: " + guild_id + " Not Found", 1009)
            }

            if (guild.ownerId == xuid) {
                return genErrorResponse("Owner guild can\'t leave. But can delete the guild", 1003)
            }

            const guildMember = await prisma.guildMember.delete({
                where: {
                    memberId: xuid,
                }
            })

            if (!guildMember) {
                return genErrorResponse("Failed to leave guild " + guild.name, 1004)
            }

            await prisma.user.update({
                data: {
                    guildId: null
                },
                where: {
                    xuid
                }
            })

            response.result = guildMember
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });

    server.delete<{ Body: DeleteGuild }>('/guild/delete', {}, async (request, reply) => {
        var response: any = defaultResponseApi()
        try {
            const xuid = request.body.xuid

            const owner = await prisma.user.findFirst({ where: { xuid } })
            if (!owner) {
                return genErrorResponse("User XUID: " + xuid + " Not Found", 1001)
            }

            if (!owner?.guildId) {
                return genErrorResponse("User XUID: " + xuid + " Don\'t have any guild", 1002)
            }

            const guild_id = owner.guildId

            const guild = await prisma.guild.findFirst({
                include: {
                    GuildMember: true,
                },
                where: {
                    guildId: guild_id
                }
            })
            if (!guild) {
                return genErrorResponse("Guild ID: " + guild_id + " Not Found", 1009)
            }

            if (guild.ownerId != request.body.xuid) {
                return genErrorResponse("This XUID not an owner of guild " + guild.name, 1003)
            }

            const deleteGuild = await prisma.guild.delete({
                where: {
                    guildId: guild_id
                }
            })

            await prisma.user.update({
                data: {
                    money: owner?.money! + ((guild.GuildMember.length * cost_member) / 2)
                },
                where: {
                    xuid
                }
            })

            await prisma.user.updateMany({
                data: {
                    guildId: null,
                },
                where: {
                    guildId: guild_id
                }
            })

            response.result = deleteGuild
            return response
        } catch (error) {
            return reply.send(genInternalErrorResponse(response, error));
        }
    });
};

export default GuildRoute;