"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LevelDB = void 0;
const chainedBatch_1 = require("./chainedBatch");
const iterator_1 = require("./iterator");
const binding = require('../binding');
const debug = require('debug')('leveldb');
globalThis.levelDbOpened = globalThis.levelDbOpened || new Set();
async function delay(ms) {
    return await new Promise(resolve => setTimeout(resolve, ms));
}
class LevelDB {
    context;
    path;
    options;
    status;
    constructor(path, options = {}) {
        this.context = binding.db_init();
        this.path = path;
        this.options = options;
        this.status = 'closed';
    }
    /**
     * Opens the database.
     * @returns {Promise} Resolves when the database has been opened.
     */
    async open() {
        if (globalThis.levelDbOpened.has(this.path)) {
            throw new Error('DB already has an open context, did you close it properly?');
        }
        globalThis.levelDbOpened.add(this.path);
        await delay(100); // (mostly) unnoticeable hack to fix race bugs in bindings (#1)
        return await new Promise((res, rej) => {
            binding.db_open(this.context, this.path, this.options, (err) => {
                if (err) {
                    debug('[leveldb] Failed to open db ', this.path, this.options, err);
                    rej(Error(err));
                }
                else {
                    this.status = 'open';
                    debug('[leveldb] DB was opened at: ', this.path);
                    globalThis.levelDbOpened.add(this.path);
                    res(true);
                }
            });
        });
    }
    isOpen() {
        return this.status === 'open';
    }
    /**
     * Closes the database.
     * @returns {Promise} Resolves when the database has been opened.
     */
    async close() {
        if (!this.isOpen()) {
            this.status = 'closed';
            globalThis.levelDbOpened.delete(this.path);
            return;
        }
        return await new Promise((res, rej) => binding.db_close(this.context, (err) => {
            if (err) {
                rej(Error(err));
            }
            else {
                this.status = 'closed';
                globalThis.levelDbOpened.delete(this.path);
                res(true);
            }
        }));
    }
    static serializeKey(key) {
        return Buffer.isBuffer(key) ? key : String(key);
    }
    static serializeValue(value) {
        return Buffer.isBuffer(value) ? value : String(value);
    }
    async put(key, value, options = {}) {
        if (this.status !== 'open')
            throw new Error('DB is not open');
        if (!key.length)
            throw new Error('Empty key');
        return await new Promise((res, rej) => {
            binding.db_put(this.context, key, value, options, err => err ? rej(Error(err)) : res(true));
        });
    }
    async get(key, options = {}) {
        if (this.status !== 'open')
            throw new Error('DB is not open');
        if (!key.length)
            throw new Error('Empty key');
        return await new Promise((res, rej) => {
            binding.db_get(this.context, key, options, (err, val) => {
                if (err) {
                    if (err.message.includes('NotFound')) {
                        return res(null);
                    }
                    return rej(new Error(err));
                }
                else {
                    return res(val);
                }
            });
        });
    }
    async getAsString(key, options = {}) {
        return String(await this.get(key, options));
    }
    async delete(key, options = {}) {
        return await new Promise((res, rej) => {
            binding.db_del(this.context, key, options, err => err ? rej(Error(err)) : res(true));
        });
    }
    chainedBatch() {
        return new chainedBatch_1.ChainedBatch(this);
    }
    async batch(operations, options = {}) {
        return await new Promise((res, rej) => {
            binding.batch_do(this.context, operations, options, err => err ? rej(Error(err)) : res(true));
        });
    }
    async approximateSize(start, end) {
        if (start == null ||
            end == null ||
            typeof start === 'function' ||
            typeof end === 'function') {
            throw new Error('approximateSize() requires valid `start` and `end` arguments');
        }
        start = LevelDB.serializeKey(start);
        end = LevelDB.serializeKey(end);
        return await new Promise((res, rej) => binding.db_approximate_size(this.context, start, end, (err, ...a) => err ? rej(Error(err)) : res(...a)));
    }
    async compactRange(start, end) {
        if (start == null || end == null || typeof start === 'function' || typeof end === 'function') {
            throw new Error('compactRange() requires valid `start` and `end` arguments');
        }
        start = LevelDB.serializeKey(start);
        end = LevelDB.serializeKey(end);
        return await new Promise((res, rej) => binding.db_compact_range(this.context, start, end, (err, ...a) => err ? rej(Error(err)) : res(...a)));
    }
    getProperty(property) {
        return binding.db_get_property(this.context, property);
    }
    /**
     * Creates a new iterator with the specified options.
     * @param options Iterator options
     * @returns {Iterator}
     */
    getIterator(options = {}) {
        if (this.status !== 'open') {
            throw new Error('cannot call iterator() before open()');
        }
        iterator_1.Iterator._setupIteratorOptions(options);
        debug('iter using options', options);
        return new iterator_1.Iterator(this, options);
    }
    async *[Symbol.asyncIterator]() {
        const it = this.getIterator();
        let next;
        while (next = await it.next()) {
            const [value, key] = next;
            yield [key, value];
        }
    }
    /**
     * Delete all entries or a range.
     */
    async clear(options) {
        return await new Promise((res, rej) => binding.clear(this.context, options, err => err ? rej(Error(err)) : res(true)));
    }
    /**
     * destroy() is used to completely remove an existing LevelDB database directory. You can use this function in place of a full directory rm if you want to be sure to only remove LevelDB-related files. If the directory only contains LevelDB files, the directory itself will be removed as well. If there are additional, non-LevelDB files in the directory, those files, and the directory, will be left alone.
     * The callback will be called when the destroy operation is complete, with a possible error argument.
     */
    static async destroy(location) {
        return await new Promise((res, rej) => binding.destroy_db(location, err => err ? rej(Error(err)) : res(true)));
    }
    /**
     * repair() can be used to attempt a restoration of a damaged LevelDB store. From the LevelDB documentation:
     *
     * If a DB cannot be opened, you may attempt to call this method to resurrect as much of the contents of the database as possible. Some data may be lost, so be careful when calling this function on a database that contains important information.
     *
     * You will find information on the repair operation in the LOG file inside the store directory.
     *
     * A repair() can also be used to perform a compaction of the LevelDB log into table files.
     */
    static async repair(location) {
        return await new Promise((res, rej) => binding.repair_db(location, err => err ? rej(Error(err)) : res(true)));
    }
}
exports.LevelDB = LevelDB;
