/// <reference types="node" />
import { ChainedBatch } from './chainedBatch';
import { Iterator } from './iterator';
export interface LevelDBOptions {
    bufferKeys?: boolean;
    snapshots?: boolean;
    permanence?: boolean;
    seek?: boolean;
    clear?: boolean;
    createIfMissing?: boolean;
    errorIfExists?: boolean;
    additionalMethods?: {
        approximateSize?: boolean;
        compactRange?: boolean;
    };
}
export interface Operation {
    type: 'del' | 'put';
    key: Buffer | string;
    value?: Buffer | string;
}
export interface OpOpts {
    sync?: boolean;
}
export interface IterOpts {
    reverse?: boolean;
    keys?: boolean;
    values?: boolean;
    fillCache?: boolean;
    keyAsBuffer?: boolean;
    valueAsBuffer?: boolean;
    limit?: number;
    highWaterMark?: boolean;
    end?: any;
    lt?: any;
    lte?: any;
    gt?: any;
    gte?: any;
}
export interface ClearOpts {
    gt?: any;
    gte?: any;
    lt?: any;
    lte?: any;
    reverse: boolean;
    limit: number;
}
export declare class LevelDB {
    context: any;
    path: string;
    options: LevelDBOptions;
    status: string;
    constructor(path: string, options?: LevelDBOptions);
    /**
     * Opens the database.
     * @returns {Promise} Resolves when the database has been opened.
     */
    open(): Promise<unknown>;
    isOpen(): boolean;
    /**
     * Closes the database.
     * @returns {Promise} Resolves when the database has been opened.
     */
    close(): Promise<unknown>;
    static serializeKey(key: any): string | Buffer;
    static serializeValue(value: any): string | Buffer;
    put(key: Buffer | string, value: Buffer | string, options?: OpOpts): Promise<boolean>;
    get(key: Buffer | string, options?: OpOpts): Promise<Buffer | null>;
    getAsString(key: Buffer | string, options?: OpOpts): Promise<string>;
    delete(key: Buffer | string, options?: OpOpts): Promise<boolean>;
    chainedBatch(): ChainedBatch;
    batch(operations: Operation[], options?: OpOpts): Promise<boolean>;
    approximateSize(start: any, end: any): Promise<unknown>;
    compactRange(start: any, end: any): Promise<unknown>;
    getProperty(property: any): any;
    /**
     * Creates a new iterator with the specified options.
     * @param options Iterator options
     * @returns {Iterator}
     */
    getIterator(options?: IterOpts): Iterator;
    [Symbol.asyncIterator](): AsyncGenerator<any[], void, unknown>;
    /**
     * Delete all entries or a range.
     */
    clear(options: ClearOpts): Promise<unknown>;
    /**
     * destroy() is used to completely remove an existing LevelDB database directory. You can use this function in place of a full directory rm if you want to be sure to only remove LevelDB-related files. If the directory only contains LevelDB files, the directory itself will be removed as well. If there are additional, non-LevelDB files in the directory, those files, and the directory, will be left alone.
     * The callback will be called when the destroy operation is complete, with a possible error argument.
     */
    static destroy(location: any): Promise<unknown>;
    /**
     * repair() can be used to attempt a restoration of a damaged LevelDB store. From the LevelDB documentation:
     *
     * If a DB cannot be opened, you may attempt to call this method to resurrect as much of the contents of the database as possible. Some data may be lost, so be careful when calling this function on a database that contains important information.
     *
     * You will find information on the repair operation in the LOG file inside the store directory.
     *
     * A repair() can also be used to perform a compaction of the LevelDB log into table files.
     */
    static repair(location: string): Promise<unknown>;
}
