"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const client_1 = __importDefault(require("@prisma/client"));
const utils_1 = require("../utils/utils");
const settings_1 = require("../constants/settings");
const GuildRoute = (server, options) => __awaiter(void 0, void 0, void 0, function* () {
    var _a, _b;
    const cost_member = Number((_b = (_a = (yield utils_1.prisma.settings.findFirst({ where: { name: settings_1.SETTINGS.GUILD_CREATE_PRICE } }))) === null || _a === void 0 ? void 0 : _a.value) !== null && _b !== void 0 ? _b : 20000);
    server.get('/guild/cost_member', {}, (request, reply) => __awaiter(void 0, void 0, void 0, function* () {
        var response = (0, utils_1.defaultResponseApi)();
        try {
            response.result = { cost_member };
            return response;
        }
        catch (error) {
            return reply.send((0, utils_1.genInternalErrorResponse)(response, error));
        }
    }));
    server.get('/guilds', {}, (request, reply) => __awaiter(void 0, void 0, void 0, function* () {
        var response = (0, utils_1.defaultResponseApi)();
        try {
            const guild = yield utils_1.prisma.guild.findMany({
                where: { isPublic: true, },
                orderBy: { updatedAt: 'desc' }
            });
            if (!guild) {
                return (0, utils_1.genErrorResponse)("Guild Empty");
            }
            response.result = guild;
            return response;
        }
        catch (error) {
            return reply.send((0, utils_1.genInternalErrorResponse)(response, error));
        }
    }));
    server.get('/guild/:id', {}, (request, reply) => __awaiter(void 0, void 0, void 0, function* () {
        var response = (0, utils_1.defaultResponseApi)();
        try {
            const guild = yield utils_1.prisma.guild.findFirst({
                where: { guildId: Number(request.params.id) }
            });
            if (!guild) {
                return (0, utils_1.genErrorResponse)("Guild ID " + request.params.id + " not found");
            }
            const guildMember = yield utils_1.prisma.guildMember.findMany({
                include: { member: true },
                where: { guildId: Number(request.params.id) }
            });
            if (!guildMember) {
                return (0, utils_1.genErrorResponse)("Guild Member ID " + request.params.id + " not found");
            }
            response.result = guild;
            response.result.members = guildMember;
            return response;
        }
        catch (error) {
            return reply.send((0, utils_1.genInternalErrorResponse)(response, error));
        }
    }));
    server.post('/guild/create', {}, (request, reply) => __awaiter(void 0, void 0, void 0, function* () {
        var response = (0, utils_1.defaultResponseApi)();
        try {
            const xuid = request.body.xuid;
            const name = request.body.name;
            const description = request.body.description;
            const max_member = request.body.maxMember;
            const is_public = request.body.isPublic;
            const owner = yield utils_1.prisma.user.findFirst({ where: { xuid } });
            const guild_cost = max_member * cost_member;
            if ((owner === null || owner === void 0 ? void 0 : owner.money) < guild_cost) {
                return (0, utils_1.genErrorResponse)("Owner money is not enough, need " + guild_cost, 1001);
            }
            const hasGuild = yield utils_1.prisma.guild.findFirst({
                where: { ownerId: xuid }
            });
            if (hasGuild) {
                return (0, utils_1.genErrorResponse)("Owner still have guild " + hasGuild.name + "", 1002);
            }
            const guildExist = yield utils_1.prisma.guild.findFirst({
                where: { name }
            });
            if (guildExist) {
                return (0, utils_1.genErrorResponse)("Guild with name " + guildExist.name + " already exists", 1003);
            }
            if (name.length > 15) {
                return (0, utils_1.genErrorResponse)("Max length name is 15", 1004);
            }
            const guild = yield utils_1.prisma.guild.create({
                data: {
                    ownerId: xuid,
                    name,
                    description,
                    maxMembers: max_member,
                    isPublic: is_public
                }
            });
            if (!guild) {
                return (0, utils_1.genErrorResponse)("Failed create guild", 1004);
            }
            yield utils_1.prisma.user.update({
                data: {
                    money: (owner === null || owner === void 0 ? void 0 : owner.money) - guild_cost,
                    guildId: guild.guildId
                },
                where: {
                    xuid: owner === null || owner === void 0 ? void 0 : owner.xuid
                }
            });
            yield utils_1.prisma.guildMember.create({
                data: {
                    guildId: guild.guildId,
                    memberId: xuid,
                    memberRole: client_1.default.GuildRole.LEADER
                }
            });
            response.result = guild;
            return response;
        }
        catch (error) {
            return reply.send((0, utils_1.genInternalErrorResponse)(response, error));
        }
    }));
    server.put('/guild', {}, (request, reply) => __awaiter(void 0, void 0, void 0, function* () {
        var response = (0, utils_1.defaultResponseApi)();
        try {
            const xuid = request.body.xuid;
            const name = request.body.name;
            const description = request.body.description;
            const max_member = request.body.maxMember;
            const is_public = request.body.isPublic;
            const userUpdater = yield utils_1.prisma.user.findFirst({ where: { xuid } });
            if (!userUpdater) {
                return (0, utils_1.genErrorResponse)("User XUID: " + xuid + " Not Found", 1001);
            }
            if (!(userUpdater === null || userUpdater === void 0 ? void 0 : userUpdater.guildId)) {
                return (0, utils_1.genErrorResponse)("User XUID: " + xuid + " Don\'t have any guild", 1002);
            }
            const guild_id = userUpdater.guildId;
            const guild = yield utils_1.prisma.guild.findFirst({
                include: {
                    GuildMember: true
                },
                where: {
                    guildId: guild_id
                }
            });
            const guild_member = yield utils_1.prisma.guildMember.findFirst({ where: { memberId: xuid } });
            if (!guild) {
                return (0, utils_1.genErrorResponse)("Guild ID: " + guild_id + " Not Found", 1009);
            }
            if ((guild_member === null || guild_member === void 0 ? void 0 : guild_member.memberRole) == client_1.default.GuildRole.MEMBER) {
                return (0, utils_1.genErrorResponse)("XUID: " + xuid + " don't have access to edit this guild", 1003);
            }
            var maxChange = null;
            var diffMax = 0;
            if (max_member != null && guild.maxMembers < max_member) {
                maxChange = "+";
                diffMax = max_member - guild.maxMembers;
                var guild_cost = diffMax * cost_member;
                if ((userUpdater === null || userUpdater === void 0 ? void 0 : userUpdater.money) < guild_cost) {
                    return (0, utils_1.genErrorResponse)("User money is not enough to add guild max member", 1004);
                }
            }
            else if (max_member != null && guild.maxMembers > max_member) {
                maxChange = "-";
                diffMax = guild.maxMembers - max_member;
                const guildLength = guild.GuildMember.length;
                if (guildLength > max_member) {
                    return (0, utils_1.genErrorResponse)("Failed update max member. Guild length is " + guildLength + " but you requesting max member to " + max_member + " ", 1005);
                }
            }
            if ((name !== null && name !== void 0 ? name : guild.name).length > 15) {
                return (0, utils_1.genErrorResponse)("Max length name is 15", 1006);
            }
            const guildUpdate = {
                name: name !== null && name !== void 0 ? name : guild.name,
                description: description !== null && description !== void 0 ? description : guild.description,
                maxMembers: max_member !== null && max_member !== void 0 ? max_member : guild.maxMembers,
                isPublic: is_public !== null && is_public !== void 0 ? is_public : guild.isPublic
            };
            const update = yield utils_1.prisma.guild.update({
                data: guildUpdate,
                where: {
                    guildId: guild_id
                }
            });
            if (maxChange == "+") {
                yield utils_1.prisma.user.update({
                    data: {
                        money: (userUpdater === null || userUpdater === void 0 ? void 0 : userUpdater.money) - (diffMax * cost_member)
                    },
                    where: {
                        xuid: userUpdater === null || userUpdater === void 0 ? void 0 : userUpdater.xuid
                    }
                });
            }
            else if (maxChange == "-") {
                yield utils_1.prisma.user.update({
                    data: {
                        money: (userUpdater === null || userUpdater === void 0 ? void 0 : userUpdater.money) + ((diffMax * cost_member) / 2)
                    },
                    where: {
                        xuid: userUpdater === null || userUpdater === void 0 ? void 0 : userUpdater.xuid
                    }
                });
            }
            response.result = update;
            return response;
        }
        catch (error) {
            return reply.send((0, utils_1.genInternalErrorResponse)(response, error));
        }
    }));
    server.post('/guild/join', {}, (request, reply) => __awaiter(void 0, void 0, void 0, function* () {
        var _c, _d;
        var response = (0, utils_1.defaultResponseApi)();
        try {
            const xuid = request.body.xuid;
            const guild_id = request.body.guildId;
            const guild = yield utils_1.prisma.guild.findFirst({
                include: {
                    GuildMember: true
                },
                where: {
                    guildId: guild_id
                }
            });
            const member = yield utils_1.prisma.user.findFirst({ where: { xuid } });
            if (member === null || member === void 0 ? void 0 : member.guildId) {
                return (0, utils_1.genErrorResponse)("Cannot join. User still have guild with ID: " + (member === null || member === void 0 ? void 0 : member.guildId), 1001);
            }
            if (!guild) {
                return (0, utils_1.genErrorResponse)("Guild ID: " + guild_id + " Not Found", 1009);
            }
            if (((_d = (_c = guild === null || guild === void 0 ? void 0 : guild.GuildMember) === null || _c === void 0 ? void 0 : _c.length) !== null && _d !== void 0 ? _d : 0) == (guild === null || guild === void 0 ? void 0 : guild.maxMembers)) {
                return (0, utils_1.genErrorResponse)("Guild " + guild.name + " has reached max member", 1002);
            }
            const guildMember = yield utils_1.prisma.guildMember.create({
                data: {
                    guildId: guild_id,
                    memberId: xuid,
                }
            });
            if (!guildMember) {
                return (0, utils_1.genErrorResponse)("Failed to join guild " + guild.name, 1003);
            }
            yield utils_1.prisma.user.update({
                data: {
                    guildId: guild.guildId
                },
                where: {
                    xuid
                }
            });
            response.result = guildMember;
            return response;
        }
        catch (error) {
            return reply.send((0, utils_1.genInternalErrorResponse)(response, error));
        }
    }));
    server.post('/guild/promote', {}, (request, reply) => __awaiter(void 0, void 0, void 0, function* () {
        var response = (0, utils_1.defaultResponseApi)();
        try {
            const xuid = request.body.xuid;
            const guild_id = request.body.guildId;
            const role = request.body.role;
            const guild = yield utils_1.prisma.guild.findFirst({
                include: {
                    GuildMember: true
                },
                where: {
                    guildId: guild_id
                }
            });
            if (!guild) {
                return (0, utils_1.genErrorResponse)("Guild ID: " + guild_id + " Not Found", 1001);
            }
            console.log({ xuid, guild_id, role });
            if (guild.ownerId == xuid) {
                return (0, utils_1.genErrorResponse)("Owner cannot demote", 1002);
            }
            const guildMember = yield utils_1.prisma.guildMember.update({
                data: {
                    memberRole: role
                },
                where: {
                    guildId: guild_id,
                    memberId: xuid,
                }
            });
            if (!guildMember) {
                return (0, utils_1.genErrorResponse)("Failed to promote ID: " + xuid, 1002);
            }
            response.result = guildMember;
            return response;
        }
        catch (error) {
            return reply.send((0, utils_1.genInternalErrorResponse)(response, error));
        }
    }));
    server.post('/guild/leave', {}, (request, reply) => __awaiter(void 0, void 0, void 0, function* () {
        var response = (0, utils_1.defaultResponseApi)();
        try {
            const xuid = request.body.xuid;
            const member = yield utils_1.prisma.user.findFirst({ where: { xuid } });
            if (!member) {
                return (0, utils_1.genErrorResponse)("User ID: " + xuid + " Not Exist", 1001);
            }
            if (!(member === null || member === void 0 ? void 0 : member.guildId)) {
                return (0, utils_1.genErrorResponse)("Cannot leave. User don\'t have any guild", 1002);
            }
            const guild_id = member === null || member === void 0 ? void 0 : member.guildId;
            const guild = yield utils_1.prisma.guild.findFirst({
                where: {
                    guildId: guild_id
                }
            });
            if (!guild) {
                return (0, utils_1.genErrorResponse)("Guild ID: " + guild_id + " Not Found", 1009);
            }
            if (guild.ownerId == xuid) {
                return (0, utils_1.genErrorResponse)("Owner guild can\'t leave. But can delete the guild", 1003);
            }
            const guildMember = yield utils_1.prisma.guildMember.delete({
                where: {
                    memberId: xuid,
                }
            });
            if (!guildMember) {
                return (0, utils_1.genErrorResponse)("Failed to leave guild " + guild.name, 1004);
            }
            yield utils_1.prisma.user.update({
                data: {
                    guildId: null
                },
                where: {
                    xuid
                }
            });
            response.result = guildMember;
            return response;
        }
        catch (error) {
            return reply.send((0, utils_1.genInternalErrorResponse)(response, error));
        }
    }));
    server.delete('/guild/delete', {}, (request, reply) => __awaiter(void 0, void 0, void 0, function* () {
        var response = (0, utils_1.defaultResponseApi)();
        try {
            const xuid = request.body.xuid;
            const owner = yield utils_1.prisma.user.findFirst({ where: { xuid } });
            if (!owner) {
                return (0, utils_1.genErrorResponse)("User XUID: " + xuid + " Not Found", 1001);
            }
            if (!(owner === null || owner === void 0 ? void 0 : owner.guildId)) {
                return (0, utils_1.genErrorResponse)("User XUID: " + xuid + " Don\'t have any guild", 1002);
            }
            const guild_id = owner.guildId;
            const guild = yield utils_1.prisma.guild.findFirst({
                include: {
                    GuildMember: true,
                },
                where: {
                    guildId: guild_id
                }
            });
            if (!guild) {
                return (0, utils_1.genErrorResponse)("Guild ID: " + guild_id + " Not Found", 1009);
            }
            if (guild.ownerId != request.body.xuid) {
                return (0, utils_1.genErrorResponse)("This XUID not an owner of guild " + guild.name, 1003);
            }
            const deleteGuild = yield utils_1.prisma.guild.delete({
                where: {
                    guildId: guild_id
                }
            });
            yield utils_1.prisma.user.update({
                data: {
                    money: (owner === null || owner === void 0 ? void 0 : owner.money) + ((guild.GuildMember.length * cost_member) / 2)
                },
                where: {
                    xuid
                }
            });
            yield utils_1.prisma.user.updateMany({
                data: {
                    guildId: null,
                },
                where: {
                    guildId: guild_id
                }
            });
            response.result = deleteGuild;
            return response;
        }
        catch (error) {
            return reply.send((0, utils_1.genInternalErrorResponse)(response, error));
        }
    }));
});
exports.default = GuildRoute;
